-- AutoStrawSwath
--
-- @author  Shaydera
-- @history     v1.0.0.0 - 2024-11-29 - Initial version
--
-- @Descripion: Automatically activates the straw swath of a combine harvester when a suitable crop type is harvested.
-- Copyright (C) Shaydera, All Rights Reserved.

local modName = g_currentModName

AutoStrawSwath = {}

AutoStrawSwath.SPEC_TABLE_NAME = "spec_"..modName..".autoStrawSwath"

---Checks if all prerequisite specializations are loaded
-- @param table specializations specializations
-- @return boolean hasPrerequisite true if all prerequisite specializations are loaded
function AutoStrawSwath.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Combine, specializations)
end

---Registers vehicle event listeners to the given vehicle type
-- Vehicle.lua defines a list of possible event listeners that will be called during runtime
-- @param table vehicleType the vehicle type
function AutoStrawSwath.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", AutoStrawSwath)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", AutoStrawSwath)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", AutoStrawSwath)
    SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", AutoStrawSwath)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", AutoStrawSwath)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", AutoStrawSwath)
end

---Registers custom specialization functions to the vehicle type
-- We need to add all custom functions. Otherwise they won't be available
-- @param table vehicleType the vehicle type
function AutoStrawSwath.registerFunctions(vehicleType)
end

---Initializes the specialization
-- Used to register xml config schema and xml savegame schema data
function AutoStrawSwath.initSpecialization()
end

---Called on loading
-- @param table savegame savegame
function AutoStrawSwath:onLoad(savegame)
    local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
    spec.dirtyFlag = self:getNextDirtyFlag()
    spec.ignoreFruitType = 0
end

---Reads the specializations data on object initialization
-- This function will normally be called on the client once after the object was loaded
-- @param int streamId the id of the network stream
-- @param table connection the network connection
function AutoStrawSwath:onReadStream(streamId, connection)
    local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
    spec.ignoreFruitType = streamReadInt8(streamId)
end

---Write the specializations data on object initialization
-- This function will normally be called on the server once after the object was loaded
-- @param int streamId the id of the network stream
-- @param table connection the network connection
function AutoStrawSwath:onWriteStream(streamId, connection)
    local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
    streamWriteInt8(streamId, spec.ignoreFruitType)
end

---Read the specializations data on object update
-- @param int streamId the id of the network stream
-- @param int timestamp the update timestamp
-- @param table connection the network connection
function AutoStrawSwath:onReadUpdateStream(streamId, timestamp, connection)
    if connection:getIsServer() then
        if streamReadBool(streamId) then
            local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
            spec.ignoreFruitType = streamReadInt8(streamId)
        end
    end
end

---Write the specializations data on object update
-- @param int streamId the id of the network stream
-- @param table connection the network connection
-- @param int dirtyMask the update dirtymask
function AutoStrawSwath:onWriteUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then
        local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
        if streamWriteBool(streamId, bitAND(dirtyMask, spec.dirtyFlag) ~= 0) then
            streamWriteInt8(streamId, spec.ignoreFruitType)
        end
    end
end

---Called on update
-- @param float dt time since last call in ms
-- @param boolean isActiveForInput true if vehicle is active for input
-- @param boolean isActiveForInputIgnoreSelection
-- @param boolean isSelected true if vehicle is selected
function AutoStrawSwath:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self[AutoStrawSwath.SPEC_TABLE_NAME]
    local isTurnedOn = self:getIsTurnedOn()
    if isTurnedOn then
        if self.isServer and self.spec_combine.swath.isAvailable then
            local fillUnitIndex = self.spec_combine.bufferFillUnitIndex or self.spec_combine.fillUnitIndex
            local fruitType = g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(self:getFillUnitFillType(fillUnitIndex))
            if fruitType ~= nil and fruitType ~= FruitType.UNKNOWN and fruitType ~= spec.ignoreFruitType then
                if not self.spec_combine.isSwathActive then
                    local fruitDesc = g_fruitTypeManager:getFruitTypeByIndex(fruitType)
                    if fruitDesc.hasWindrow then
                        self:setIsSwathActive(true)
                    end
                end
                spec.ignoreFruitType = fruitType
                self:raiseDirtyFlags(spec.dirtyFlag)
            end
        end
    end
end
